<?php
/**
 * Anonymous Watch Video
 *
 * @package     AutomatorWP\Integrations\Presto_Player\Triggers\Anonymous_Watch_Video
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Presto_Player_Anonymous_Watch_Video extends AutomatorWP_Integration_Trigger {

    public $integration = 'presto_player';
    public $trigger = 'presto_player_anonymous_watch_video';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'anonymous'         => true,
            'label'             => __( 'Guest fully watches a video', 'automatorwp-pro' ),
            'select_option'     => __( 'Guest fully watches a <strong>video</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Post Title. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'Guest fully watches %1$s %2$s time(s)', 'automatorwp-pro' ), '{post}', '{times}' ),
            /* translators: %1$s: Post Title. */
            'log_label'         => sprintf( __( 'Guest fully watches %1$s', 'automatorwp-pro' ), '{post}' ),
            'action'            => 'presto_player_progress',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 2,
            'options'           => array(
                'post' => automatorwp_utilities_post_option( array(
                    'name' => __( 'Video:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'any video', 'automatorwp-pro' ),
                    'post_type' => 'pp_video_block'
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_presto_player_video_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int $video_id
     * @param int $percent
     */
    public function listener( $video_id, $percent ) {

        $user_id = get_current_user_id();

        // Bail if user is logged in
        if( $user_id !== 0 ) {
            return;
        }

        // Bail if user not watched the video entirely
        if( $percent < 100 ) {
            return;
        }

        automatorwp_trigger_event( array(
            'trigger'           => $this->trigger,
            'video_id'          => $video_id,
        ) );

    }

    /**
     * Anonymous deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function anonymous_deserves_trigger( $deserves_trigger, $trigger, $event, $trigger_options, $automation ) {

        // Don't deserve if video is not received
        if( ! isset( $event['video_id'] ) ) {
            return false;
        }

        $video_id = absint( $event['video_id'] );

        // Don't deserve if video doesn't exists
        if( $video_id === 0 ) {
            return false;
        }

        $model = new \PrestoPlayer\Models\Video();

        if( ! $model ) {
            return '';
        }

        $video = $model->get( $video_id );
        
        $video_post_id = $video->__get( 'post_id' );

        // Don't deserve if video doesn't match with the trigger option
        if( $trigger_options['post'] !== 'any' && absint( $video_post_id ) !== absint( $trigger_options['post'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_anonymous_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['video_id'] = ( isset( $event['video_id'] ) ? $event['video_id'] : 0 );

        return $log_meta;

    }

}

new AutomatorWP_Presto_Player_Anonymous_Watch_Video();